<?php
/**
 * Elementor Montoya Counter Widget.
 *
 * Elementor widget that enables you to add an animated numbered counter to your page
 *
 * @since 1.0.0
 */
class Elementor_Montoya_Counter_Widget extends \Elementor\Widget_Base {

	/**
	 * Get widget name.
	 *
	 * Retrieve Counter widget name.
	 *
	 * @since 1.0.0
	 * @access public
	 *
	 * @return string Widget name.
	 */
	public function get_name() {
		return 'montoya_counter';
	}

	/**
	 * Get widget title.
	 *
	 * Retrieve Counter widget title.
	 *
	 * @since 1.0.0
	 * @access public
	 *
	 * @return string Widget title.
	 */
	public function get_title() {
		return __( 'Counter', 'montoya-elementor-widgets' );
	}

	/**
	 * Get widget icon.
	 *
	 * Retrieve Counter widget icon.
	 *
	 * @since 1.0.0
	 * @access public
	 *
	 * @return string Widget icon.
	 */
	public function get_icon() {
		return 'eicon-counter-circle';
	}

	/**
	 * Get widget categories.
	 *
	 * Retrieve the list of categories the Counter widget belongs to.
	 *
	 * @since 1.0.0
	 * @access public
	 *
	 * @return array Widget categories.
	 */
	public function get_categories() {
		return [ 'montoya-widgets' ];
	}

	/**
	 * Register Counter widget controls.
	 *
	 * Adds different input fields to allow the user to change and customize the widget settings.
	 *
	 * @since 1.0.0
	 * @access protected
	 */
	protected function register_controls() {

		$this->start_controls_section(
			'content_section',
			[
				'label' => __( 'Content', 'montoya-elementor-widgets' ),
				'tab' => \Elementor\Controls_Manager::TAB_CONTENT,
			]
		);

		$this->add_control(
			'data_start',
			[
				'label' => __( 'Start Value', 'montoya-elementor-widgets' ),
				'type' => \Elementor\Controls_Manager::NUMBER,
				'show_label' => true,
				'default' => 1000
			]
		);

		$this->add_control(
			'data_target',
			[
				'label' => __( 'Target Value', 'montoya-elementor-widgets' ),
				'type' => \Elementor\Controls_Manager::NUMBER,
				'show_label' => true,
				'default' => 3000
			]
		);
		
		$this->add_control(
			'data_symbol',
			[
				'label' => __( 'Data Symbol (example %, +)', 'montoya-elementor-widgets' ),
				'type' => \Elementor\Controls_Manager::TEXT,
				'show_label' => true,
				'default' => ''
			]
		);

		$this->add_control(
			'text_size',
			[
				'label'        => __( 'Text Size', 'montoya-elementor-widgets' ),
				'type'         => \Elementor\Controls_Manager::SELECT,
				'default'      => 'h1',
				'options'      => array( 'h1' => __( 'H1', 'montoya-elementor-widgets' ),
										'h2' => __( 'H2', 'montoya-elementor-widgets' ),
									 	'h3' => __( 'H3', 'montoya-elementor-widgets' ),
										'h4' => __( 'H4', 'montoya-elementor-widgets' ),
										'h5' => __( 'H5', 'montoya-elementor-widgets' ),
										'h6' => __( 'H6', 'montoya-elementor-widgets' ))
			]
		);

		$this->add_control(
			'has_animation',
			[
				'label' => __( 'Has Animation', 'montoya-elementor-widgets' ),
				'type' => \Elementor\Controls_Manager::SWITCHER,
				'label_on' => __( 'Yes', 'montoya-elementor-widgets' ),
				'label_off' => __( 'No', 'montoya-elementor-widgets' ),
				'return_value' => 'yes',
				'default' => 'no',
			]
		);

		$this->add_control(
			'animation_delay',
			[
				'label' => __( 'Animation Delay', 'montoya-elementor-widgets' ),
				'type' => \Elementor\Controls_Manager::NUMBER,
				'show_label' => true,
				'default' => 0
			]
		);

		$this->end_controls_section();

	}

	/**
	 * Render Counter widget output on the frontend.
	 *
	 * Written in PHP and used to generate the final HTML.
	 *
	 * @since 1.0.0
	 * @access protected
	 */
	protected function render() {

		$settings = $this->get_settings_for_display();

		$montoya_animation = $settings['has_animation'];
		if( $montoya_animation == 'no'){

			$montoya_animation = false;
		}

		echo '<' . esc_attr( $settings['text_size'] ) . ' class="number-counter no-margins';
		if( $montoya_animation ){

			echo ' has-animation';
		}
		echo '"';
		if( $montoya_animation ){

			echo ' data-delay="'. esc_attr( $settings['animation_delay'] ) . '"';
		}
		echo ' data-target="' . esc_attr( $settings['data_target'] ) . '" data-symbol="' . esc_attr( $settings['data_symbol'] ) . '">' . esc_attr( $settings['data_start'] ) . '</' . esc_attr( $settings['text_size'] ) . '>';
		
	}

}

?>
