<?php
/**
 * Elementor Montoya Image Clipped Widget.
 *
 * Gradually clips the image on scroll
 *
 * @since 1.0.0
 */
class Elementor_Montoya_Image_Clipped_Widget extends \Elementor\Widget_Base {

	/**
	 * Get widget name.
	 *
	 * Retrieve image clipped widget name.
	 *
	 * @since 1.0.0
	 * @access public
	 *
	 * @return string Widget name.
	 */
	public function get_name() {
		return 'montoya_image_clipped';
	}

	/**
	 * Get widget title.
	 *
	 * Retrieve image clipped widget title.
	 *
	 * @since 1.0.0
	 * @access public
	 *
	 * @return string Widget title.
	 */
	public function get_title() {
		return __( 'Image Clipped', 'montoya-elementor-widgets' );
	}

	/**
	 * Get widget icon.
	 *
	 * Retrieve image clipped widget icon.
	 *
	 * @since 1.0.0
	 * @access public
	 *
	 * @return string Widget icon.
	 */
	public function get_icon() {
		return 'eicon-grow';
	}

	/**
	 * Get widget categories.
	 *
	 * Retrieve the list of categories the image clipped widget belongs to.
	 *
	 * @since 1.0.0
	 * @access public
	 *
	 * @return array Widget categories.
	 */
	public function get_categories() {
		return [ 'montoya-widgets' ];
	}

	/**
	 * Register image clipped widget controls.
	 *
	 * Adds different input fields to allow the user to change and customize the widget settings.
	 *
	 * @since 1.0.0
	 * @access protected
	 */
	protected function register_controls() {

		$this->start_controls_section(
			'content_section',
			[
				'label' => __( 'Content', 'montoya-elementor-widgets' ),
				'tab' => \Elementor\Controls_Manager::TAB_CONTENT,
			]
		);

		$this->add_control(
			'image_clipped',
			[
				'label' => __( 'The Image', 'montoya-elementor-widgets' ),
				'type' => \Elementor\Controls_Manager::MEDIA,
				'default' => [
					'url' => \Elementor\Utils::get_placeholder_image_src(),
				]
			]
		);
		
		$this->add_control(
			'video_clipped',
			[
				'label' => __( 'The Video', 'montoya-elementor-widgets' ),
				'description' => __( 'Self-hosted video URL (usually in .mp4 format) to have a clipped video instead of just image.', 'montoya-elementor-widgets' ),
				'type' => \Elementor\Controls_Manager::TEXT,
				'label_block' => true,
				'show_label' => true,
			]
		);
		
		$this->add_control(
			'content_width',
			[
				'label'        => __( 'Content Width', 'montoya-elementor-widgets' ),
				'type'         => Elementor\Controls_Manager::SELECT,
				'default'      => 'content-full-width',
				'options'      => array( 'content-full-width' => __( 'Full', 'montoya-elementor-widgets' ),
										'content-max-width' => __( 'Boxed', 'montoya-elementor-widgets' ) ),
				'prefix_class' => ''
			]
		);
		
		$this->add_control(
			'image_clipped_text',
			[
				'label' => __( 'The Text (HTML allowed)', 'montoya-elementor-widgets' ),
				'description' => __( 'This the content displayed after has been fully opened on scroll.', 'montoya-elementor-widgets' ),
				'type' => \Elementor\Controls_Manager::CODE,
				'language' => 'html',
				'rows' => 20,
			]
		);
		
		$this->add_control(
			'image_clipped_text_align',
			[
				'label'        => 'Text Alignment',
				'type'         => Elementor\Controls_Manager::SELECT,
				'default'      => 'text-align-center',
				'options'      => array( 'text-align-center' => __( 'Center', 'montoya-elementor-widgets' ),
										'text-align-left' => __( 'Left', 'montoya-elementor-widgets' ),
										'text-align-right' => __( 'Right', 'montoya-elementor-widgets' ) ),
				'prefix_class' => ''
			]
		);

		$this->end_controls_section();

	}

	/**
	 * Renders image clipped widget output on the frontend.
	 *
	 * Written in PHP and used to generate the final HTML.
	 *
	 * @since 1.0.0
	 * @access protected
	 */
	protected function render() {

		$settings = $this->get_settings_for_display();

		$montoya_elementor_img_alt = trim( strip_tags( get_post_meta( $settings['image_clipped']['id'], '_wp_attachment_image_alt', true ) ) );
		$montoya_elementor_img_caption = wp_get_attachment_caption( $settings['image_clipped']['id'] );
				
		echo '<div class="clipped-image-wrapper">';
		echo '<div class="clipped-image-pin">';
		echo '<div class="clipped-image">';
		echo '<img src="' . esc_url( $settings['image_clipped']['url'] ) . '" alt="' . esc_attr( $montoya_elementor_img_alt ) . '">';
		
		if( !empty( trim( $settings['video_clipped'] ) ) ){

			echo '<div class="content-video-wrapper">';
			echo '<video loop muted playsinline class="bgvid">';
			echo '<source src="' . esc_url( $settings['video_clipped'] ) . '" type="video/mp4">';
			echo '</video>';
			echo '</div>';
		}
											
		echo '<div class="clipped-image-gradient"></div>';
		echo '</div>';
		echo '</div>';
		echo '<div class="clipped-image-content ' . esc_attr( $settings['content_width'] ) . ' ' . esc_attr( $settings['image_clipped_text_align'] ) . '">';
        echo wp_kses_post( $settings['image_clipped_text'] );
		echo '</div>';
		echo '</div>';
	}

}

?>
