<?php
/**
 * Elementor Montoya Clients Moving Gallery Widget.
 *
 * Elementor widget that inserts a moving gallery on scroll displaying client images attached to a link opening in a new tab.
 *
 * @since 1.0.0
 */
class Elementor_Montoya_Clients_Moving_Gallery_Widget extends \Elementor\Widget_Base {

	/**
	 * Get widget name.
	 *
	 * Retrieve Clients Moving Gallery widget name.
	 *
	 * @since 1.0.0
	 * @access public
	 *
	 * @return string Widget name.
	 */
	public function get_name() {
		return 'montoya_clients_moving_gallery';
	}

	/**
	 * Get widget title.
	 *
	 * Retrieve Clients Moving Gallery widget title.
	 *
	 * @since 1.0.0
	 * @access public
	 *
	 * @return string Widget title.
	 */
	public function get_title() {
		return __( 'Clients Moving Gallery', 'montoya-elementor-widgets' );
	}

	/**
	 * Get widget icon.
	 *
	 * Retrieve clients moving gallery widget icon.
	 *
	 * @since 1.0.0
	 * @access public
	 *
	 * @return string Widget icon.
	 */
	public function get_icon() {
		return 'eicon-product-images';
	}

	/**
	 * Get widget categories.
	 *
	 * Retrieve the list of categories the clients moving gallery widget belongs to.
	 *
	 * @since 1.0.0
	 * @access public
	 *
	 * @return array Widget categories.
	 */
	public function get_categories() {
		return [ 'montoya-widgets' ];
	}

	/**
	 * Register Clients Moving Gallery widget controls.
	 *
	 * Adds different input fields to allow the user to change and customize the widget settings.
	 *
	 * @since 1.0.0
	 * @access protected
	 */
	protected function register_controls() {

		$this->start_controls_section(
			'content_section',
			[
				'label' => __( 'Content', 'montoya-elementor-widgets' ),
				'tab' => \Elementor\Controls_Manager::TAB_CONTENT,
			]
		);

		$repeater = new \Elementor\Repeater();

		$repeater->add_control(
			'logo',
			[
				'label' => __( 'Client Logo', 'montoya-elementor-widgets' ),
				'type' => \Elementor\Controls_Manager::MEDIA,
				'default' => [
					'url' => \Elementor\Utils::get_placeholder_image_src(),
				]
			]
		);

		$repeater->add_control(
			'name',
			[
				'label' => __( 'Client Name', 'montoya-elementor-widgets' ),
				'type' => \Elementor\Controls_Manager::TEXT,
				'show_label' => true,
			]
		);
		
		$repeater->add_control(
			'url',
			[
				'label' => __( 'Client Link URL', 'montoya-elementor-widgets' ),
				'type' => \Elementor\Controls_Manager::TEXT,
				'show_label' => true,
			]
		);

		$this->add_control(
			'clients',
			[
				'label' => __( 'Clients', 'montoya-elementor-widgets' ),
				'type' => \Elementor\Controls_Manager::REPEATER,
				'fields' => $repeater->get_controls(),
				'title_field' => '{{{ name }}}',
			]
		);
		
		$this->add_control(
			'direction',
			[
				'label'        => __( 'Direction', 'montoya-elementor-widgets' ),
				'type'         => \Elementor\Controls_Manager::SELECT,
				'default'      => 'fw-gallery',
				'options'      => array( 'fw-gallery' => __( 'Forward', 'montoya-elementor-widgets' ),
										'bw-gallery' => __( 'Backward', 'montoya-elementor-widgets' ) )
			]
		);
		
		$this->end_controls_section();

	}

	/**
	 * Render Moving Gallery widget output on the frontend.
	 *
	 * Written in PHP and used to generate the final HTML.
	 *
	 * @since 1.0.0
	 * @access protected
	 */
	protected function render() {

		$settings = $this->get_settings_for_display();

		echo '<div class="moving-gallery ' . esc_attr( $settings['direction'] ) . '">';
		echo '<ul class="wrapper-gallery">';
		foreach ( $settings['clients'] as $client ) {

			$image_alt = get_post_meta( $client['logo']['id'], '_wp_attachment_image_alt', TRUE );

			echo '<li class="link">';
			if( !empty( $client['url'] ) ){

				echo '<a target="_blank" href="' . esc_url( $client['url'] ) . '">';
			}
			echo '<img src="' . esc_url( $client['logo']['url'] ) . '" alt="client image" />';
			if( !empty( $client['name'] ) ){
		
				echo '<div class="moving-gallery-caption">' . wp_kses_post( $client['name'] ) . '</div>';
			}
			if( !empty( $client['url'] ) ){

				echo '</a>';
			}
			echo '</li>';

		}
		echo '</ul>';
		echo '</div>';
		
	}

}

?>
